#! /usr/bin/python
#
# Copyright (c) 2009 Andreas Wetzel
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License version 2 as
# published by the Free Software Foundation.
#
# This code is heavily based on the migration script written in Ruby by
# Ralph Juhnke. See CREDITS file for details.
#

########## MIGRATION SCRIPT CONFIGURATION ##########

BUGZILLA_DB = {
	"host":   "host",
        "db":     "bugzilla",
        "user":   "user",
        "passwd": "pass",
}

REDMINE_DB =  {
	"host":   "host",
	"db":     "redmine",
        "user":   "user",
        "passwd": "pass",
}

ATTACHMENT_PATH = "/var/lib/redmine/files"

ENABLED_TRACKERS = [ 1, 2]		# 1: bug tracker, 2: feature tracker, ...
ENABLED_MODULES  = [
	'issue_tracking',
#	'time_tracking',
#	'news',
#	'documents',
#	'files',
#	'wiki',
#	'repository',
#	'boards',
]

PRIORITY_MAP = {       
	"P1": 7,
	"P2": 6,
	"P3": 4,
	"P4": 3,
	"P5": 3,
}

TRACKER_MAP = {
        "feature change request": 2,
	"enhancement": 2,
	"trivial":     1,
	"normal":      1,
	"minor":       1,
	"major":       1,
        "critical":    1,
	"blocker":     1,
}

ISSUE_STATUS_MAP = {
	"UNCONFIRMED": 1,
	"NEW":         1,
	"VERIFIED":    2,
	"ASSIGNED":    2,
	"RESOLVED":    5,
	"CLOSED":      5,
	"REOPENED":    7,
}


########## MIGRATION SCRIPT CONFIGURATION ##########

import os
from logging import *
import MySQLdb

basicConfig(level=INFO)

class DB:
	def __init__( self, settings):
		debug( "connecting to db '%s' on %s..." % (settings["db"], settings["host"]))
		self.settings = settings
		self.db = MySQLdb.connect( host=settings["host"], db=settings["db"], user=settings["user"], passwd=settings["passwd"])
		self.cursor = self.db.cursor()
		info( "connected to db '%s' on %s" % (settings["db"], settings["host"]))

	def execsql( self, sql):
		debug( "execute SQL: DB=%s, sql='%s'" % (self.settings["db"], sql))
		return self.cursor.execute( sql)

	def select( self, sql):
		self.execsql( sql)
		rows = self.cursor.fetchall()
		return rows

	def __del__( self):
		self.cursor.close()
		self.db.commit()
		self.db.close()
		info( "disconnected from db '%s' on %s" % (self.settings["db"], self.settings["host"]))


class bz2redmine:
	def db_connect( self):
		self.db_bz = DB( BUGZILLA_DB)
		self.db_rm = DB( REDMINE_DB)

	def db_disconnect( self):
		del self.db_bz
		del self.db_rm
## HELPERS

	def find_min_created_at_for_product(self, product_id):
		when = '1970-01-01 10:22:25'
		for row in self.db_bz.select( "SELECT MIN(b.creation_ts) FROM products p JOIN bugs b ON b.product_id = p.id WHERE product_id=%d" % product_id):
			when = row[0]
		return when

	def find_max_bug_when_for_product(self, product_id):
		when = '1970-01-01 10:22:25'
		sql = "SELECT MAX(l.bug_when) FROM products p JOIN bugs b ON b.product_id = p.id JOIN longdescs l ON l.bug_id = b.bug_id WHERE b.product_id=%d" % product_id
		for row in self.db_bz.select( sql):
			when = row[0]
		return when

	def find_max_bug_when( self, bug_id):
		bug_when = '1970-01-01 10:22:25'
		for row in self.db_bz.select( "SELECT MAX(bug_when) FROM longdescs WHERE bug_id=%d" % bug_id):
			bug_when = row[0]
		return bug_when

	def find_version_id( self, project_id, version):
		result = -1
		for row in self.db_rm.select( "SELECT id FROM versions WHERE project_id=%d AND name='%s'" % (project_id, version)):
			result = row[0]
		return result

## MIGRATION

	def clean_redmine_db( self):
		tables = [
			# projects
			"projects", "projects_trackers", "enabled_modules",

			# versions
			"versions",

			# users
    			"users", "user_preferences", "members", "messages", "tokens", "watchers",

			# issues
			"issues", "journals",

			# issue relations
			"issue_relations",

			# attachments
			"attachments",

			# members
			"members",

			# others
			"boards",
			"custom_fields_projects",
			"documents",
			"news",
			"queries",
			"repositories",
			"time_entries",
			"wiki_content_versions",
			"wiki_contents",
			"wiki_pages",
			"wiki_redirects",
			"wikis",
		]
		for table in tables:
			self.db_rm.execsql( "DELETE FROM %s" % table)

	def migrate_projects( self):
		info( "migrating projects");
		for row in self.db_bz.select( "SELECT products.id, products.name, products.description, products.classification_id, products.disallownew, classifications.name as classification_name FROM products, classifications WHERE products.classification_id = classifications.id"):
			identifier = row[1].lower().replace( " ", "-").replace( ".", "")
			status = 9 if (row[3] == 1) else 1
			created_at = self.find_min_created_at_for_product( row[0])
			updated_at = self.find_max_bug_when_for_product( row[0])
			self.db_rm.execsql( "INSERT INTO projects (id, name, description, is_public, identifier, created_on, updated_on, status) values (%d, '%s', '%s', %d, '%s', '%s', '%s', %d)" %
                                                                   (row[0], row[1], row[2],  1,       identifier, created_at, updated_at, status))

			# enable project trackers
			for tracker_id in ENABLED_TRACKERS:
				self.db_rm.execsql("INSERT INTO projects_trackers (project_id, tracker_id) VALUES (%d, %d)" % (row[0], tracker_id))

			# enable project modules
			for module in ENABLED_MODULES:
      				self.db_rm.execsql("INSERT INTO enabled_modules (project_id, name) VALUES (%d, '%s')" % (row[0], module))


	def migrate_versions( self):
		info( "migrating versions");
		for project_id, version in self.db_bz.select( "SELECT product_id AS project_id, value AS name FROM versions"):
			self.db_rm.execsql( "INSERT INTO versions (project_id, name) VALUES (%d, '%s')" % (project_id, version))


	def migrate_users( self):
		info( "migrating users");
		for userid, login, realname, disabledtext in self.db_bz.select( "SELECT userid, login_name, realname, disabledtext FROM profiles"):
			firstname = lastname = ""
			if realname:
				if " " in realname:
					firstname, lastname = realname.split( " ", 1);
				else: firstname = realname
			if not firstname: firstname = "Foo"
			if not lastname: lastname = "Bar"
			firstname = MySQLdb.escape_string( firstname);
			lastname = MySQLdb.escape_string( lastname);

			status = 1 if not disabledtext else 3
			self.db_rm.execsql( "INSERT INTO users (id, login, mail, firstname, lastname, language, mail_notification, status) values (%d, '%s', '%s', \"%s\", '%s', '%s', %d, '%s')" %
							       (userid, login, login, firstname, lastname, 'en', 0,                status))
			prefs = """---
:comments_sorting: asc
:no_self_notified: true
"""
			self.db_rm.execsql("INSERT INTO user_preferences (user_id,others) values (%d, '%s')" % (userid, prefs))

	def migrate_issues( self):
		info( "migrating issues")

		current_bug_id = -1
		sql = "SELECT bugs.bug_id, bugs.assigned_to, bugs.bug_status, bugs.creation_ts, bugs.short_desc, bugs.product_id, bugs.reporter, bugs.version, bugs.resolution, bugs.estimated_time, bugs.remaining_time, bugs.deadline, bugs.bug_severity, bugs.priority, bugs.component_id, bugs.status_whiteboard AS whiteboard, bugs.bug_file_loc AS url, longdescs.thetext, longdescs.bug_when, longdescs.who, longdescs.isprivate FROM bugs, longdescs WHERE bugs.bug_id = longdescs.bug_id ORDER BY bugs.creation_ts, longdescs.bug_when"


		for bug in self.db_bz.select( sql):
      			bug_id, assigned_to, bug_status, creation_ts, short_desc, product_id, reporter, version, resolution, estimated_time, remaining_time, deadline, bug_severity, priority, component_id, whiteboard, url, thetext, bug_when, who, isprivate = bug
			short_desc = MySQLdb.escape_string( short_desc)
			thetext = MySQLdb.escape_string( thetext)

			if(current_bug_id != bug_id):
				status_id = 1
				version_id = self.find_version_id( product_id, version)
				updated_at = self.find_max_bug_when( bug_id)
				priority_id = PRIORITY_MAP[priority]
				tracker_id = TRACKER_MAP[bug_severity]
				status_id = ISSUE_STATUS_MAP[bug_status]

				self.db_rm.execsql( "INSERT INTO issues (id, project_id, subject, description, assigned_to_id, author_id, created_on, updated_on, start_date, estimated_hours, priority_id, fixed_version_id, category_id, tracker_id, status_id) values (%d, %d, '%s', '%s', %d, %d, '%s', '%s', '%s', '%s', %d, %d, %d, %d, %d)" %
        						                (bug_id, product_id, short_desc, thetext, assigned_to, reporter,  creation_ts, updated_at, creation_ts, estimated_time, priority_id, version_id,      component_id, tracker_id, status_id))
			        current_bug_id = bug_id
			else:
				self.db_rm.execsql( "INSERT INTO journals (journalized_id, journalized_type, user_id, notes, created_on)  VALUES (%d, '%s', %d, '%s', '%s')" %
									  (bug_id,         "Issue",          who,     thetext, bug_when))

	def migrate_issue_relations( self):
		info( "migrating issue relations")
		for blocked, dependson in self.db_bz.select( "SELECT blocked, dependson FROM dependencies"):
			self.db_rm.execsql( "INSERT INTO issue_relations (issue_from_id, issue_to_id, relation_type) values (%d, %d, '%s')" % (blocked, dependson, "blocks"))

		for dupe_of, dupe in self.db_bz.select( "SELECT dupe_of, dupe FROM duplicates"):
			self.db_rm.execsql( "INSERT INTO issue_relations (issue_from_id, issue_to_id, relation_type) values (%d, %d, '%s')" % (dupe_of, dupe, "duplicates"))

	def migrate_attachments( self):
		try:
			attachments = self.db_bz.select( "SELECT attach_id, bug_id, filename, mimetype, submitter_id, creation_ts, description, thedata FROM attachments")
		except:
			# seems to be a newer bugzilla with "thedata" in separate table
			attachments = self.db_bz.select( "SELECT attachments.attach_id, attachments.bug_id, attachments.filename, attachments.mimetype, attachments.submitter_id, attachments.creation_ts, attachments.description, attach_data.thedata FROM attachments, attach_data WHERE attachments.attach_id = attach_data.id")
		
		for attach_id, bug_id, filename, mimetype, submitter_id, creation_ts, description, thedata in attachments:
			description = MySQLdb.escape_string( description)

			# create a unique filename
			disk_filename = "attachment_%d%s" % (attach_id, os.path.splitext( filename)[1])
      			filesize = len( thedata)
			f = open( ATTACHMENT_PATH + os.sep + disk_filename, "wb")
			f.write( thedata)
			f.close()

			self.db_rm.execsql( "INSERT INTO attachments (id, container_id, container_type, filename, filesize, disk_filename, content_type, digest, downloads, author_id, created_on, description) values (%d, %d, '%s', '%s', %d, '%s', '%s', '%s', %d, %d, '%s', '%s')" %
								     (attach_id, bug_id, 'Issue',       filename, filesize, disk_filename, mimetype,     '',     0,         submitter_id, creation_ts, description))

	def setup_tweaks( self):
		info( "applying local tweaks")

		# put your local tweaks here

		# set projects non-public and active
		# self.db_rm.execsql( "UPDATE projects SET is_public=0, status=1")

	def migrate( self):
		self.db_connect()
		self.clean_redmine_db()
		self.migrate_projects()
		self.migrate_versions()
		self.migrate_users()
		self.migrate_issues()
		self.migrate_issue_relations()
		self.migrate_attachments()
		self.setup_tweaks()
		self.db_disconnect()
		

b2r = bz2redmine()
b2r.migrate()
